<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class SettingsController extends Controller
{
    public function index()
    {
        $settings = [
            'company' => Setting::getByGroup('company'),
            'payment' => Setting::getByGroup('payment'),
            'discount' => Setting::getByGroup('discount'),
            'general' => Setting::getByGroup('general'),
        ];

        return view('admin.settings.index', compact('settings'));
    }

    public function update(Request $request)
    {
        $validated = $request->validate([
            'company_name' => 'nullable|string|max:255',
            'company_email' => 'nullable|email',
            'company_phone' => 'nullable|string|max:50',
            'company_address' => 'nullable|string',
            'company_logo' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'installation_discount' => 'nullable|numeric|min:0|max:100',
            'enable_installation_discount' => 'nullable',
            'mpesa_enabled' => 'nullable',
            'mpesa_till_number' => 'nullable|string',
            'mpesa_consumer_key' => 'nullable|string',
            'mpesa_consumer_secret' => 'nullable|string',
            'mpesa_passkey' => 'nullable|string',
            'mpesa_environment' => 'nullable|in:sandbox,production',
            'bank_enabled' => 'nullable',
            'bank_name' => 'nullable|string',
            'bank_account_name' => 'nullable|string',
            'bank_account_number' => 'nullable|string',
            'bank_branch' => 'nullable|string',
            'currency' => 'nullable|string|max:10',
            'currency_symbol' => 'nullable|string|max:10',
            'quotation_validity_days' => 'nullable|integer|min:1',
            'default_terms' => 'nullable|string',
        ]);

        // Handle logo upload
        if ($request->hasFile('company_logo')) {
            $oldLogo = Setting::getValue('company_logo');
            if ($oldLogo) {
                Storage::disk('public')->delete($oldLogo);
            }
            $validated['company_logo'] = $request->file('company_logo')->store('settings', 'public');
        }

        foreach ($validated as $key => $value) {
            if ($value !== null) {
                $type = $this->getSettingType($key);
                Setting::setValue($key, $value, $type);
            }
        }

        return back()->with('success', 'Settings updated successfully.');
    }

    private function getSettingType(string $key): string
    {
        $booleanKeys = ['enable_installation_discount', 'mpesa_enabled', 'bank_enabled'];
        $numberKeys = ['installation_discount', 'quotation_validity_days'];

        if (in_array($key, $booleanKeys)) {
            return 'boolean';
        }

        if (in_array($key, $numberKeys)) {
            return 'number';
        }

        return 'string';
    }
}
