<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Services\OrderProcessingService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class OrderProcessingController extends Controller
{
    protected OrderProcessingService $orderProcessingService;

    public function __construct(OrderProcessingService $orderProcessingService)
    {
        $this->orderProcessingService = $orderProcessingService;
    }

    /**
     * Handle incoming email webhook
     * POST /api/orders/webhook/email
     */
    public function processEmailWebhook(Request $request)
    {
        // Verify webhook signature
        if (!$this->verifyEmailWebhook($request)) {
            Log::warning('Invalid email webhook signature', [
                'ip' => $request->ip(),
            ]);
            return response()->json(['error' => 'Unauthorized'], 401);
        }

        // Extract email content - adjust based on your email provider's webhook format
        // Common providers: SendGrid, Mailgun, Postmark
        $content = $this->extractEmailContent($request);
        $senderEmail = $this->extractSenderEmail($request);

        // Verify sender is authorized
        $authorizedEmail = config('services.order_processing.admin_email', 'info@onespotsolutionsltd.co.ke');
        if (strtolower($senderEmail) !== strtolower($authorizedEmail)) {
            Log::warning('Unauthorized email sender for order processing', [
                'sender' => $senderEmail,
                'authorized' => $authorizedEmail,
            ]);
            return response()->json(['error' => 'Unauthorized sender'], 403);
        }

        // Process the message
        $result = $this->orderProcessingService->handleMessage($content, 'email');

        Log::info('Email webhook processed', [
            'sender' => $senderEmail,
            'content' => substr($content, 0, 100),
            'result' => $result,
        ]);

        return response()->json($result, $result['success'] ? 200 : 400);
    }

    /**
     * Handle incoming WhatsApp webhook
     * POST /api/orders/webhook/whatsapp
     */
    public function processWhatsAppWebhook(Request $request)
    {
        // Verify webhook signature
        if (!$this->verifyWhatsAppWebhook($request)) {
            Log::warning('Invalid WhatsApp webhook signature', [
                'ip' => $request->ip(),
            ]);
            return response()->json(['error' => 'Unauthorized'], 401);
        }

        // Extract WhatsApp message content - adjust based on your provider (Twilio, Africa's Talking, etc.)
        $content = $this->extractWhatsAppContent($request);
        $senderNumber = $this->extractWhatsAppSender($request);

        // Normalize phone number (remove + and spaces)
        $senderNumber = preg_replace('/[^0-9]/', '', $senderNumber);
        $authorizedNumber = preg_replace('/[^0-9]/', '', config('services.order_processing.whatsapp_admin', '254743456809'));

        // Verify sender is authorized
        if ($senderNumber !== $authorizedNumber) {
            Log::warning('Unauthorized WhatsApp sender for order processing', [
                'sender' => $senderNumber,
                'authorized' => $authorizedNumber,
            ]);
            return response()->json(['error' => 'Unauthorized sender'], 403);
        }

        // Process the message
        $result = $this->orderProcessingService->handleMessage($content, 'whatsapp');

        Log::info('WhatsApp webhook processed', [
            'sender' => $senderNumber,
            'content' => substr($content, 0, 100),
            'result' => $result,
        ]);

        return response()->json($result, $result['success'] ? 200 : 400);
    }

    /**
     * WhatsApp webhook verification endpoint (GET request)
     * Required by some WhatsApp providers for initial webhook setup
     */
    public function verifyWhatsAppWebhook(Request $request)
    {
        // Meta/Facebook WhatsApp Business API verification
        $mode = $request->query('hub_mode');
        $token = $request->query('hub_verify_token');
        $challenge = $request->query('hub_challenge');

        $verifyToken = config('services.order_processing.whatsapp_webhook_secret');

        if ($mode === 'subscribe' && $token === $verifyToken) {
            Log::info('WhatsApp webhook verified');
            return response($challenge, 200);
        }

        return response('Forbidden', 403);
    }

    /**
     * Verify email webhook signature
     */
    protected function verifyEmailWebhook(Request $request): bool
    {
        $secret = config('services.order_processing.email_webhook_secret');
        
        // If no secret configured, allow all (for development)
        if (empty($secret)) {
            return true;
        }

        // Check for common webhook signature headers
        // SendGrid: X-Twilio-Email-Event-Webhook-Signature
        // Mailgun: X-Mailgun-Signature
        // Postmark: X-Postmark-Signature
        
        $signature = $request->header('X-Webhook-Signature') 
            ?? $request->header('X-Mailgun-Signature')
            ?? $request->header('X-Postmark-Signature')
            ?? $request->input('signature');

        if (!$signature) {
            // For development/testing, check for secret in query or body
            return $request->input('secret') === $secret 
                || $request->query('secret') === $secret;
        }

        // Verify HMAC signature
        $payload = $request->getContent();
        $expectedSignature = hash_hmac('sha256', $payload, $secret);

        return hash_equals($expectedSignature, $signature);
    }

    /**
     * Verify WhatsApp webhook signature
     */
    protected function verifyWhatsAppWebhookSignature(Request $request): bool
    {
        $secret = config('services.order_processing.whatsapp_webhook_secret');
        
        // If no secret configured, allow all (for development)
        if (empty($secret)) {
            return true;
        }

        // Check for signature header (varies by provider)
        $signature = $request->header('X-Hub-Signature-256') 
            ?? $request->header('X-Twilio-Signature')
            ?? $request->input('signature');

        if (!$signature) {
            return $request->input('secret') === $secret 
                || $request->query('secret') === $secret;
        }

        // Verify HMAC signature
        $payload = $request->getContent();
        $expectedSignature = 'sha256=' . hash_hmac('sha256', $payload, $secret);

        return hash_equals($expectedSignature, $signature);
    }

    /**
     * Extract email content from webhook payload
     * Adjust based on your email provider
     */
    protected function extractEmailContent(Request $request): string
    {
        // SendGrid inbound parse
        if ($request->has('text')) {
            return $request->input('text', '');
        }

        // Mailgun
        if ($request->has('body-plain')) {
            return $request->input('body-plain', '');
        }

        // Postmark
        if ($request->has('TextBody')) {
            return $request->input('TextBody', '');
        }

        // Generic/custom
        if ($request->has('content')) {
            return $request->input('content', '');
        }

        // Check subject line as fallback
        if ($request->has('subject')) {
            return $request->input('subject', '');
        }

        return $request->input('message', '');
    }

    /**
     * Extract sender email from webhook payload
     */
    protected function extractSenderEmail(Request $request): string
    {
        return $request->input('from') 
            ?? $request->input('sender') 
            ?? $request->input('From')
            ?? $request->input('email')
            ?? '';
    }

    /**
     * Extract WhatsApp message content
     * Adjust based on your WhatsApp provider
     */
    protected function extractWhatsAppContent(Request $request): string
    {
        // Twilio
        if ($request->has('Body')) {
            return $request->input('Body', '');
        }

        // Africa's Talking
        if ($request->has('text')) {
            return $request->input('text', '');
        }

        // Meta WhatsApp Business API
        $entry = $request->input('entry.0.changes.0.value.messages.0');
        if ($entry && isset($entry['text']['body'])) {
            return $entry['text']['body'];
        }

        // Generic
        return $request->input('message') 
            ?? $request->input('content')
            ?? '';
    }

    /**
     * Extract WhatsApp sender number
     */
    protected function extractWhatsAppSender(Request $request): string
    {
        // Twilio
        if ($request->has('From')) {
            return str_replace('whatsapp:', '', $request->input('From', ''));
        }

        // Africa's Talking
        if ($request->has('from')) {
            return $request->input('from', '');
        }

        // Meta WhatsApp Business API
        $entry = $request->input('entry.0.changes.0.value.messages.0');
        if ($entry && isset($entry['from'])) {
            return $entry['from'];
        }

        return $request->input('sender') ?? '';
    }
}
