<?php

namespace App\Http\Controllers\Shop;

use App\Http\Controllers\Controller;
use App\Models\CartItem;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CartController extends Controller
{
    /**
     * Display the cart contents.
     */
    public function index()
    {
        $cartItems = CartItem::with(['product.images', 'product.category'])
            ->where('user_id', auth()->id())
            ->get();

        $subtotal = $cartItems->sum('subtotal');
        $installationTotal = $cartItems->sum('installation_cost');
        $total = $subtotal + $installationTotal;

        return view('shop.cart', compact('cartItems', 'subtotal', 'installationTotal', 'total'));
    }

    /**
     * Add a product to the cart.
     */
    public function add(Request $request)
    {
        $validated = $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'nullable|integer|min:1|max:100',
            'include_installation' => 'nullable|boolean',
        ]);

        $product = Product::findOrFail($validated['product_id']);
        $quantity = $validated['quantity'] ?? 1;
        $includeInstallation = $validated['include_installation'] ?? false;

        // Check stock
        if (!$product->in_stock || $product->stock_quantity < $quantity) {
            return back()->with('error', 'Sorry, this product is out of stock or insufficient quantity available.');
        }

        // Check if item already in cart
        $existingItem = CartItem::where('user_id', auth()->id())
            ->where('product_id', $product->id)
            ->first();

        if ($existingItem) {
            $newQuantity = $existingItem->quantity + $quantity;
            if ($newQuantity > $product->stock_quantity) {
                return back()->with('error', 'Cannot add more items. Maximum stock limit reached.');
            }
            $existingItem->update([
                'quantity' => $newQuantity,
                'include_installation' => $includeInstallation,
            ]);
        } else {
            CartItem::create([
                'user_id' => auth()->id(),
                'product_id' => $product->id,
                'quantity' => $quantity,
                'include_installation' => $includeInstallation && $product->requires_installation,
            ]);
        }

        return back()->with('success', 'Product added to cart successfully!');
    }

    /**
     * Update cart item quantity.
     */
    public function update(Request $request, CartItem $item)
    {
        // Ensure user owns this cart item
        if ($item->user_id !== auth()->id()) {
            abort(403);
        }

        $validated = $request->validate([
            'quantity' => 'required|integer|min:1|max:100',
            'include_installation' => 'nullable|boolean',
        ]);

        // Check stock
        if ($item->product->stock_quantity < $validated['quantity']) {
            return back()->with('error', 'Requested quantity exceeds available stock.');
        }

        $item->update([
            'quantity' => $validated['quantity'],
            'include_installation' => $validated['include_installation'] ?? $item->include_installation,
        ]);

        return back()->with('success', 'Cart updated successfully!');
    }

    /**
     * Remove item from cart.
     */
    public function remove(CartItem $item)
    {
        // Ensure user owns this cart item
        if ($item->user_id !== auth()->id()) {
            abort(403);
        }

        $item->delete();

        return back()->with('success', 'Item removed from cart.');
    }

    /**
     * Clear entire cart.
     */
    public function clear()
    {
        CartItem::where('user_id', auth()->id())->delete();

        return back()->with('success', 'Cart cleared successfully.');
    }

    /**
     * Get cart item count (for AJAX).
     */
    public function count()
    {
        $count = 0;
        if (auth()->check()) {
            $count = CartItem::where('user_id', auth()->id())->sum('quantity');
        }

        return response()->json(['count' => $count]);
    }
}
