<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\SoftDeletes;

class Order extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'order_number',
        'user_id',
        'status',
        'payment_status',
        'payment_method',
        'customer_name',
        'customer_email',
        'customer_phone',
        'shipping_address',
        'city',
        'county',
        'subtotal',
        'shipping_cost',
        'discount',
        'total',
        'includes_installation',
        'installation_cost',
        'installation_discount',
        'notes',
        'processed_at',
        'processed_via',
    ];

    protected $casts = [
        'subtotal' => 'decimal:2',
        'shipping_cost' => 'decimal:2',
        'discount' => 'decimal:2',
        'total' => 'decimal:2',
        'installation_cost' => 'decimal:2',
        'installation_discount' => 'decimal:2',
        'includes_installation' => 'boolean',
        'processed_at' => 'datetime',
    ];

    /**
     * User who placed the order
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Order items
     */
    public function items(): HasMany
    {
        return $this->hasMany(OrderItem::class);
    }

    /**
     * Payments for this order
     */
    public function payments(): HasMany
    {
        return $this->hasMany(Payment::class);
    }

    /**
     * Latest payment
     */
    public function latestPayment(): HasOne
    {
        return $this->hasOne(Payment::class)->latest();
    }

    /**
     * Generate unique order number
     */
    public static function generateOrderNumber(): string
    {
        $prefix = 'OSS'; // OneSpot Shop
        $date = now()->format('Ymd');
        $random = strtoupper(substr(uniqid(), -4));
        return "{$prefix}-{$date}-{$random}";
    }

    /**
     * Check if order is paid
     */
    public function getIsPaidAttribute(): bool
    {
        return $this->payment_status === 'paid';
    }

    /**
     * Order status badges with colors
     */
    public function getStatusBadgeAttribute(): array
    {
        $badges = [
            'pending' => ['color' => 'warning', 'label' => 'Pending'],
            'processing' => ['color' => 'info', 'label' => 'Processing'],
            'confirmed' => ['color' => 'primary', 'label' => 'Confirmed'],
            'shipped' => ['color' => 'secondary', 'label' => 'Shipped'],
            'delivered' => ['color' => 'success', 'label' => 'Delivered'],
            'cancelled' => ['color' => 'danger', 'label' => 'Cancelled'],
        ];

        return $badges[$this->status] ?? ['color' => 'secondary', 'label' => $this->status];
    }

    /**
     * Payment status badges
     */
    public function getPaymentStatusBadgeAttribute(): array
    {
        $badges = [
            'pending' => ['color' => 'warning', 'label' => 'Pending'],
            'paid' => ['color' => 'success', 'label' => 'Paid'],
            'failed' => ['color' => 'danger', 'label' => 'Failed'],
            'refunded' => ['color' => 'info', 'label' => 'Refunded'],
        ];

        return $badges[$this->payment_status] ?? ['color' => 'secondary', 'label' => $this->payment_status];
    }

    /**
     * Scope for pending orders
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Scope for paid orders
     */
    public function scopePaid($query)
    {
        return $query->where('payment_status', 'paid');
    }

    /**
     * Scope for processing orders
     */
    public function scopeProcessing($query)
    {
        return $query->where('status', 'processing');
    }

    /**
     * Scope for cancelled orders
     */
    public function scopeCancelled($query)
    {
        return $query->where('status', 'cancelled');
    }
}
