<?php

namespace App\Services;

use App\Models\Order;
use Illuminate\Support\Facades\Log;

class WhatsAppService
{
    /**
     * Business WhatsApp number for receiving order notifications
     */
    protected string $businessPhone = '254740940303';

    /**
     * Generate wa.me link for sending WhatsApp message to customer
     *
     * @param string $phone Customer phone number
     * @param string $message Pre-filled message
     * @return string wa.me URL
     */
    public function generateCustomerLink(string $phone, string $message): string
    {
        $cleanPhone = $this->formatPhoneNumber($phone);
        $encodedMessage = urlencode($message);
        
        return "https://wa.me/{$cleanPhone}?text={$encodedMessage}";
    }

    /**
     * Generate wa.me link for admin to notify about new order
     *
     * @param string $message Pre-filled message
     * @return string wa.me URL
     */
    public function generateAdminLink(string $message): string
    {
        $encodedMessage = urlencode($message);
        
        return "https://wa.me/{$this->businessPhone}?text={$encodedMessage}";
    }

    /**
     * Generate order invoice message for customer
     *
     * @param Order $order
     * @return string
     */
    public function generateCustomerInvoiceMessage(Order $order): string
    {
        $items = $order->items->map(function($item) {
            return "- {$item->product_name} x{$item->quantity}: KSh " . number_format($item->subtotal, 2);
        })->join("\n");

        return "Hello {$order->customer_name}!\n\n" .
               "Thank you for your order with Onespot Solutions.\n\n" .
               "*Order #: {$order->order_number}*\n" .
               "*Date:* " . $order->created_at->format('M d, Y') . "\n\n" .
               "*Items:*\n{$items}\n\n" .
               "*Total: KSh " . number_format($order->total, 2) . "*\n\n" .
               "We'll notify you once your order is ready.\n\n" .
               "Thank you for choosing Onespot Solutions!";
    }

    /**
     * Generate order notification for admin
     *
     * @param Order $order
     * @return string
     */
    public function generateAdminOrderMessage(Order $order): string
    {
        return "New Order Received!\n\n" .
               "Order #: {$order->order_number}\n" .
               "Customer: {$order->customer_name}\n" .
               "Phone: {$order->customer_phone}\n" .
               "Amount: KSh " . number_format($order->total, 2) . "\n\n" .
               "Please check admin panel for details.";
    }

    /**
     * Generate WhatsApp link to send message to customer using order's phone number
     *
     * @param Order $order
     * @param string|null $customMessage Optional custom message, defaults to invoice message
     * @return string wa.me URL
     */
    public function generateCustomerLinkFromOrder(Order $order, ?string $customMessage = null): string
    {
        $phone = $order->customer_phone;
        
        // Fallback to user's phone if order phone is empty
        if (empty($phone) && $order->user) {
            $phone = $order->user->phone;
        }
        
        if (empty($phone)) {
            return '#'; // No phone available
        }
        
        $message = $customMessage ?? $this->generateCustomerInvoiceMessage($order);
        
        return $this->generateCustomerLink($phone, $message);
    }

    /**
     * Format phone number for wa.me (remove + and spaces)
     *
     * @param string $phone
     * @return string
     */
    protected function formatPhoneNumber(string $phone): string
    {
        // Remove all non-numeric characters except leading +
        $cleaned = preg_replace('/[^0-9]/', '', $phone);
        
        // If starts with 0, assume Kenya and replace with 254
        if (str_starts_with($cleaned, '0')) {
            $cleaned = '254' . substr($cleaned, 1);
        }
        
        return $cleaned;
    }

    /**
     * Legacy send method - now logs and returns link
     * Kept for backward compatibility
     *
     * @param string $phone
     * @param string $message
     * @param string|null $fileUrl
     * @return bool
     */
    public function send(string $phone, string $message, ?string $fileUrl = null): bool
    {
        Log::info("WhatsApp Message to {$phone}: {$message}");
        
        if ($fileUrl) {
            Log::info("WhatsApp Attachment to {$phone}: {$fileUrl}");
        }

        return true;
    }
}
