@extends('shop.layouts.app')

@section('title', $product->name . ' - Onespot Solutions Shop')
@section('description', Str::limit($product->short_description, 160))

@section('content')
<style>
    /* Product Detail Page Styles */
    .product-detail {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 3rem;
        margin-bottom: 4rem;
    }

    .product-gallery {
        position: sticky;
        top: 100px;
    }

    .main-image {
        background: rgba(30, 41, 59, 0.6);
        border: 1px solid rgba(255, 255, 255, 0.1);
        border-radius: 16px;
        overflow: hidden;
        aspect-ratio: 1;
        display: flex;
        align-items: center;
        justify-content: center;
        margin-bottom: 1rem;
    }

    .main-image img {
        width: 100%;
        height: 100%;
        object-fit: contain;
    }

    .main-image i {
        font-size: 5rem;
        color: #475569;
    }

    .thumbnail-grid {
        display: grid;
        grid-template-columns: repeat(4, 1fr);
        gap: 0.75rem;
    }

    .thumbnail {
        background: rgba(30, 41, 59, 0.6);
        border: 2px solid rgba(255, 255, 255, 0.1);
        border-radius: 8px;
        overflow: hidden;
        aspect-ratio: 1;
        cursor: pointer;
        transition: all 0.2s;
    }

    .thumbnail:hover,
    .thumbnail.active {
        border-color: var(--primary);
    }

    .thumbnail img {
        width: 100%;
        height: 100%;
        object-fit: cover;
    }

    .product-info h1 {
        font-size: 2rem;
        font-weight: 700;
        margin-bottom: 0.5rem;
        line-height: 1.3;
    }

    .product-meta {
        display: flex;
        flex-wrap: wrap;
        gap: 1rem;
        margin-bottom: 1.5rem;
        font-size: 0.9rem;
        color: #64748b;
    }

    .product-meta span {
        display: flex;
        align-items: center;
        gap: 0.5rem;
    }

    .product-meta i {
        color: var(--primary);
    }

    .product-pricing {
        background: rgba(30, 41, 59, 0.6);
        border: 1px solid rgba(255, 255, 255, 0.1);
        border-radius: 12px;
        padding: 1.5rem;
        margin-bottom: 1.5rem;
    }

    .price-display {
        display: flex;
        align-items: center;
        gap: 1rem;
        margin-bottom: 0.5rem;
    }

    .price-current {
        font-size: 2rem;
        font-weight: 700;
        color: var(--success);
    }

    .price-original {
        font-size: 1.25rem;
        color: #64748b;
        text-decoration: line-through;
    }

    .discount-badge {
        background: var(--danger);
        color: white;
        padding: 0.25rem 0.75rem;
        border-radius: 999px;
        font-size: 0.8rem;
        font-weight: 600;
    }

    .stock-status {
        display: flex;
        align-items: center;
        gap: 0.5rem;
        font-size: 0.9rem;
    }

    .stock-status.in-stock {
        color: var(--success);
    }

    .stock-status.out-of-stock {
        color: var(--danger);
    }

    .quantity-selector {
        display: flex;
        align-items: center;
        gap: 1rem;
        margin-bottom: 1.5rem;
    }

    .quantity-selector label {
        font-weight: 500;
        color: #94a3b8;
    }

    .quantity-input {
        display: flex;
        align-items: center;
        background: rgba(15, 23, 42, 0.8);
        border: 1px solid rgba(255, 255, 255, 0.1);
        border-radius: 8px;
        overflow: hidden;
    }

    .quantity-input button {
        width: 40px;
        height: 40px;
        background: none;
        border: none;
        color: #94a3b8;
        cursor: pointer;
        font-size: 1rem;
        transition: all 0.2s;
    }

    .quantity-input button:hover {
        background: rgba(255, 255, 255, 0.1);
        color: var(--primary);
    }

    .quantity-input input {
        width: 60px;
        height: 40px;
        text-align: center;
        background: none;
        border: none;
        border-left: 1px solid rgba(255, 255, 255, 0.1);
        border-right: 1px solid rgba(255, 255, 255, 0.1);
        color: #e2e8f0;
        font-size: 1rem;
        font-weight: 500;
    }

    .quantity-input input:focus {
        outline: none;
    }

    .add-to-cart-form {
        display: flex;
        gap: 1rem;
        margin-bottom: 1.5rem;
    }

    .add-to-cart-form .btn {
        padding: 0.875rem 2rem;
        font-size: 1rem;
    }

    .btn-cart {
        flex: 1;
        background: linear-gradient(135deg, var(--primary), #0284c7);
        color: white;
        justify-content: center;
    }

    .btn-cart:hover {
        transform: translateY(-2px);
        box-shadow: 0 10px 25px -10px rgba(14, 165, 233, 0.5);
    }

    .btn-cart:disabled {
        opacity: 0.5;
        cursor: not-allowed;
        transform: none;
    }

    .btn-wishlist {
        background: rgba(100, 116, 139, 0.2);
        color: #94a3b8;
        border: 1px solid rgba(100, 116, 139, 0.3);
    }

    .btn-wishlist:hover {
        color: var(--danger);
        border-color: rgba(239, 68, 68, 0.3);
    }

    .product-features {
        display: grid;
        grid-template-columns: repeat(2, 1fr);
        gap: 1rem;
        margin-bottom: 1.5rem;
    }

    .feature-item {
        display: flex;
        align-items: center;
        gap: 0.75rem;
        padding: 0.75rem;
        background: rgba(30, 41, 59, 0.4);
        border-radius: 8px;
    }

    .feature-item i {
        font-size: 1.25rem;
        color: var(--primary);
    }

    .feature-item span {
        font-size: 0.9rem;
        color: #94a3b8;
    }

    .product-tabs {
        margin-bottom: 1.5rem;
    }

    .tabs-header {
        display: flex;
        gap: 0.5rem;
        border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        margin-bottom: 1.5rem;
    }

    .tab-btn {
        padding: 0.75rem 1.5rem;
        background: none;
        border: none;
        color: #64748b;
        font-size: 0.95rem;
        font-weight: 500;
        cursor: pointer;
        border-bottom: 2px solid transparent;
        transition: all 0.2s;
        margin-bottom: -1px;
    }

    .tab-btn:hover {
        color: #94a3b8;
    }

    .tab-btn.active {
        color: var(--primary);
        border-bottom-color: var(--primary);
    }

    .tab-content {
        display: none;
    }

    .tab-content.active {
        display: block;
    }

    .description-content {
        color: #94a3b8;
        line-height: 1.8;
    }

    .description-content p {
        margin-bottom: 1rem;
    }

    .specifications-table {
        width: 100%;
        border-collapse: collapse;
    }

    .specifications-table tr {
        border-bottom: 1px solid rgba(255, 255, 255, 0.1);
    }

    .specifications-table td {
        padding: 0.75rem 0;
        font-size: 0.9rem;
    }

    .specifications-table td:first-child {
        color: #64748b;
        width: 40%;
    }

    .specifications-table td:last-child {
        color: #e2e8f0;
    }

    /* Related Products */
    .related-section {
        margin-top: 4rem;
    }

    .section-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 2rem;
    }

    .section-header h2 {
        font-size: 1.5rem;
        font-weight: 600;
    }

    .related-grid {
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
        gap: 1.5rem;
    }

    .product-card {
        background: rgba(30, 41, 59, 0.6);
        backdrop-filter: blur(10px);
        border: 1px solid rgba(255, 255, 255, 0.1);
        border-radius: 16px;
        overflow: hidden;
        transition: all 0.3s;
    }

    .product-card:hover {
        transform: translateY(-5px);
        border-color: rgba(14, 165, 233, 0.3);
        box-shadow: 0 20px 40px -15px rgba(0, 0, 0, 0.5);
    }

    .product-card .product-image {
        aspect-ratio: 4/3;
        background: rgba(15, 23, 42, 0.8);
        display: flex;
        align-items: center;
        justify-content: center;
        position: relative;
    }

    .product-card .product-image img {
        width: 100%;
        height: 100%;
        object-fit: cover;
    }

    .product-card .product-image i {
        font-size: 3rem;
        color: #475569;
    }

    .product-card .product-content {
        padding: 1.25rem;
    }

    .product-card .product-category {
        color: var(--primary);
        font-size: 0.8rem;
        font-weight: 500;
        margin-bottom: 0.5rem;
    }

    .product-card .product-title {
        font-size: 1rem;
        font-weight: 600;
        margin-bottom: 0.5rem;
        display: -webkit-box;
        -webkit-line-clamp: 2;
        -webkit-box-orient: vertical;
        overflow: hidden;
    }

    .product-card .product-price {
        display: flex;
        align-items: center;
        gap: 0.5rem;
        margin-bottom: 1rem;
    }

    .product-card .product-price .current {
        font-size: 1.1rem;
        font-weight: 700;
        color: var(--success);
    }

    .product-card .product-price .original {
        color: #64748b;
        text-decoration: line-through;
        font-size: 0.85rem;
    }

    /* Responsive */
    @media (max-width: 1024px) {
        .product-detail {
            grid-template-columns: 1fr;
        }

        .product-gallery {
            position: relative;
            top: 0;
        }
    }

    @media (max-width: 768px) {
        .product-info h1 {
            font-size: 1.5rem;
        }

        .price-current {
            font-size: 1.5rem;
        }

        .product-features {
            grid-template-columns: 1fr;
        }

        .add-to-cart-form {
            flex-direction: column;
        }

        .tabs-header {
            overflow-x: auto;
            -webkit-overflow-scrolling: touch;
        }
    }
</style>

<!-- Page Header -->
<div class="page-header">
    <div class="page-header-content">
        <div class="breadcrumb">
            <a href="{{ route('shop.home') }}">Home</a>
            <span>/</span>
            <a href="{{ route('shop.products') }}">Products</a>
            <span>/</span>
            @if($product->category)
                <a href="{{ route('shop.products', ['category' => $product->category_id]) }}">{{ $product->category->name }}</a>
                <span>/</span>
            @endif
            <span>{{ Str::limit($product->name, 30) }}</span>
        </div>
    </div>
</div>

<!-- Main Content -->
<div class="main-content">
    @if(session('success'))
        <div class="alert alert-success">
            <i class="fas fa-check-circle"></i> {{ session('success') }}
        </div>
    @endif

    @if(session('error'))
        <div class="alert alert-error">
            <i class="fas fa-exclamation-circle"></i> {{ session('error') }}
        </div>
    @endif

    <div class="product-detail">
        <!-- Product Gallery -->
        <div class="product-gallery">
            <div class="main-image" id="mainImage">
                @if($product->images->count() > 0)
                    <img src="{{ Storage::url($product->images->first()->path) }}" alt="{{ $product->name }}" id="mainImageSrc">
                @else
                    <i class="fas fa-camera"></i>
                @endif
            </div>
            @if($product->images->count() > 1)
                <div class="thumbnail-grid">
                    @foreach($product->images as $index => $image)
                        <div class="thumbnail {{ $index === 0 ? 'active' : '' }}" onclick="changeImage('{{ Storage::url($image->path) }}', this)">
                            <img src="{{ Storage::url($image->path) }}" alt="{{ $product->name }}">
                        </div>
                    @endforeach
                </div>
            @endif
        </div>

        <!-- Product Info -->
        <div class="product-info">
            <h1>{{ $product->name }}</h1>
            
            <div class="product-meta">
                @if($product->category)
                    <span><i class="fas fa-folder"></i> {{ $product->category->name }}</span>
                @endif
                @if($product->brand)
                    <span><i class="fas fa-tag"></i> {{ $product->brand }}</span>
                @endif
                @if($product->sku)
                    <span><i class="fas fa-barcode"></i> SKU: {{ $product->sku }}</span>
                @endif
                @if($product->model)
                    <span><i class="fas fa-cube"></i> Model: {{ $product->model }}</span>
                @endif
            </div>

            <!-- Pricing -->
            <div class="product-pricing">
                <div class="price-display">
                    @if($product->sale_price && $product->sale_price < $product->price)
                        <span class="price-current">KSh {{ number_format($product->sale_price) }}</span>
                        <span class="price-original">KSh {{ number_format($product->price) }}</span>
                        @php
                            $discount = round((($product->price - $product->sale_price) / $product->price) * 100);
                        @endphp
                        <span class="discount-badge">Save {{ $discount }}%</span>
                    @else
                        <span class="price-current">KSh {{ number_format($product->price) }}</span>
                    @endif
                </div>
                <div class="stock-status {{ $product->stock_status === 'in_stock' ? 'in-stock' : 'out-of-stock' }}">
                    @if($product->stock_status === 'in_stock')
                        <i class="fas fa-check-circle"></i>
                        <span>In Stock ({{ $product->stock_quantity }} available)</span>
                    @else
                        <i class="fas fa-times-circle"></i>
                        <span>Out of Stock</span>
                    @endif
                </div>
            </div>

            <!-- Add to Cart -->
            @auth
                <form action="{{ route('cart.add') }}" method="POST" class="add-to-cart-form">
                    @csrf
                    <input type="hidden" name="product_id" value="{{ $product->id }}">
                    
                    <div class="quantity-selector">
                        <label>Quantity:</label>
                        <div class="quantity-input">
                            <button type="button" onclick="decreaseQuantity()"><i class="fas fa-minus"></i></button>
                            <input type="number" name="quantity" id="quantity" value="1" min="1" max="{{ $product->stock_quantity }}">
                            <button type="button" onclick="increaseQuantity()"><i class="fas fa-plus"></i></button>
                        </div>
                    </div>

                    <button type="submit" class="btn btn-cart" {{ $product->stock_status !== 'in_stock' ? 'disabled' : '' }}>
                        <i class="fas fa-shopping-cart"></i>
                        {{ $product->stock_status === 'in_stock' ? 'Add to Cart' : 'Out of Stock' }}
                    </button>
                </form>
            @else
                <div class="add-to-cart-form">
                    <a href="{{ route('login') }}" class="btn btn-cart">
                        <i class="fas fa-shopping-cart"></i>
                        Login to Add to Cart
                    </a>
                </div>
            @endauth

            <!-- Features -->
            <div class="product-features">
                <div class="feature-item">
                    <i class="fas fa-truck"></i>
                    <span>Nationwide Delivery</span>
                </div>
                <div class="feature-item">
                    <i class="fas fa-shield-halved"></i>
                    <span>{{ $product->warranty ?? '1 Year' }} Warranty</span>
                </div>
                <div class="feature-item">
                    <i class="fas fa-tools"></i>
                    <span>Installation Available</span>
                </div>
                <div class="feature-item">
                    <i class="fas fa-phone"></i>
                    <span>Technical Support</span>
                </div>
            </div>

            <!-- Tabs -->
            <div class="product-tabs">
                <div class="tabs-header">
                    <button class="tab-btn active" onclick="switchTab('description')">Description</button>
                    <button class="tab-btn" onclick="switchTab('specifications')">Specifications</button>
                </div>

                <div class="tab-content active" id="description">
                    <div class="description-content">
                        @if($product->short_description)
                            <p><strong>{{ $product->short_description }}</strong></p>
                        @endif
                        @if($product->description)
                            {!! nl2br(e($product->description)) !!}
                        @else
                            <p>No detailed description available for this product.</p>
                        @endif
                    </div>
                </div>

                <div class="tab-content" id="specifications">
                    @if(is_array($product->specifications) && count($product->specifications) > 0)
                        <table class="specifications-table">
                            @foreach($product->specifications as $key => $value)
                                <tr>
                                    <td>{{ ucfirst(str_replace('_', ' ', $key)) }}</td>
                                    <td>{{ $value }}</td>
                                </tr>
                            @endforeach
                        </table>
                    @else
                        <p style="color: #64748b;">No detailed specifications available.</p>
                    @endif
                </div>
            </div>
        </div>
    </div>

    <!-- Related Products -->
    @if($relatedProducts->count() > 0)
        <div class="related-section">
            <div class="section-header">
                <h2>Related Products</h2>
                <a href="{{ route('shop.products', ['category' => $product->category_id]) }}" class="btn btn-secondary">
                    View All <i class="fas fa-arrow-right"></i>
                </a>
            </div>

            <div class="related-grid">
                @foreach($relatedProducts as $related)
                    <div class="product-card">
                        <div class="product-image">
                            @if($related->images->count() > 0)
                                <img src="{{ Storage::url($related->images->first()->path) }}" alt="{{ $related->name }}">
                            @else
                                <i class="fas fa-box"></i>
                            @endif
                        </div>
                        <div class="product-content">
                            <div class="product-category">{{ $related->category->name ?? 'Uncategorized' }}</div>
                            <h3 class="product-title">{{ $related->name }}</h3>
                            <div class="product-price">
                                @if($related->sale_price && $related->sale_price < $related->price)
                                    <span class="current">KSh {{ number_format($related->sale_price) }}</span>
                                    <span class="original">KSh {{ number_format($related->price) }}</span>
                                @else
                                    <span class="current">KSh {{ number_format($related->price) }}</span>
                                @endif
                            </div>
                            <a href="{{ route('shop.product', $related) }}" class="btn btn-primary" style="width: 100%; justify-content: center;">
                                View Details
                            </a>
                        </div>
                    </div>
                @endforeach
            </div>
        </div>
    @endif
</div>

@push('scripts')
<script>
    // Image gallery
    function changeImage(src, thumbnail) {
        document.getElementById('mainImageSrc').src = src;
        document.querySelectorAll('.thumbnail').forEach(t => t.classList.remove('active'));
        thumbnail.classList.add('active');
    }

    // Quantity controls
    function increaseQuantity() {
        const input = document.getElementById('quantity');
        const max = parseInt(input.max);
        const current = parseInt(input.value);
        if (current < max) {
            input.value = current + 1;
        }
    }

    function decreaseQuantity() {
        const input = document.getElementById('quantity');
        const current = parseInt(input.value);
        if (current > 1) {
            input.value = current - 1;
        }
    }

    // Tabs
    function switchTab(tabId) {
        document.querySelectorAll('.tab-btn').forEach(btn => btn.classList.remove('active'));
        document.querySelectorAll('.tab-content').forEach(content => content.classList.remove('active'));
        
        event.target.classList.add('active');
        document.getElementById(tabId).classList.add('active');
    }
</script>
@endpush
@endsection
